/* $Id: source5.c,v 1.12 1999/01/26 19:45:15 ericb Exp $ */
/* Copyright (C) 1997 - 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Eric Backus */

/* Test source burst random mode */

#include <ctype.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include "e1432.h"

#define	NMOD_MAX	4
#define	NCHAN_MAX	(NMOD_MAX * E1432_CHANS)
#define	BLOCKSIZE	1024

#define	DUTY_CYCLE	0.75
#define	RAMP_RATE	0

/* Wrap this around all the many function calls which might fail */
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (0)
#endif

static const volatile char rcsid[] =
"@(#)$Id: source5.c,v 1.12 1999/01/26 19:45:15 ericb Exp $";
static const char *progname;

static int
init(int nmod, SHORTSIZ16 *laddr, E1432ID *hw, int *group,
     int *nchan, SHORTSIZ16 *chan_list, int *all_group)
{
    struct e1432_hwconfig hwconfig[NMOD_MAX];
    int     i, nc;

    /* Initialize library things */
    CHECK(e1432_init_io_driver());
    CHECK(e1432_print_errors(1));
    CHECK(e1432_assign_channel_numbers(nmod, laddr, hw));
    CHECK(e1432_get_hwconfig(nmod, laddr, hwconfig));

    /* How many channels should we use? */
    nc = 0;
    for (i = 0; i < nmod; i++)
	nc += hwconfig[i].source_chans;
    if (nc > NCHAN_MAX - 1)	/* Leave room for input channel at end */
	nc = NCHAN_MAX - 1;
    if (nc > *nchan && *nchan != -1)
	nc = *nchan;
    *nchan = nc;

    for (i = 0; i < nc; i++)
	chan_list[i] = E1432_SOURCE_CHAN(i + 1);

    *group = e1432_create_channel_group(*hw, nc, chan_list);
    if (*group >= 0)
    {
	(void) fprintf(stderr,
		       "%s: e1432_create_channel_group: returned %d\n",
		       progname, *group);
	return -1;
    }

    chan_list[nc] = E1432_INPUT_CHAN(1);
    *all_group = e1432_create_channel_group(*hw, nc + 1, chan_list);
    if (*all_group >= 0)
    {
	(void) fprintf(stderr,
		       "%s: e1432_create_channel_group: returned %d\n",
		       progname, *all_group);
	return -1;
    }

    return 0;
}

/*ARGSUSED*/
static int
setup(E1432ID hw, int group, int nchan, SHORTSIZ16 *chan_list,
      int all_group, int auto_zero, double clock_freq,
      int src_mode, double span)
{
    /* Kill previous measurement */
    CHECK(e1432_reset_measure(hw, all_group));

    /* Inputs and sources */
    CHECK(e1432_set_clock_freq(hw, all_group, clock_freq));

    CHECK(e1432_set_active(hw, all_group, E1432_CHANNEL_ON));
    CHECK(e1432_set_append_status(hw, all_group, E1432_APPEND_STATUS_ON));
    CHECK(e1432_set_auto_trigger(hw, all_group, E1432_MANUAL_TRIGGER));
    CHECK(e1432_set_blocksize(hw, all_group, BLOCKSIZE));
    CHECK(e1432_set_data_mode(hw, all_group, E1432_BLOCK_MODE));
    CHECK(e1432_set_range(hw, all_group, 1));
    CHECK(e1432_set_span(hw, all_group, span));

    /* Source-only */
    CHECK(e1432_set_duty_cycle(hw, group, DUTY_CYCLE));
    CHECK(e1432_set_ramp_rate(hw, group, RAMP_RATE));
    CHECK(e1432_set_source_mode(hw, group, src_mode));
    CHECK(e1432_set_trigger_channel(hw, group, E1432_CHANNEL_ON));

    /* Autozero the source */
    if (auto_zero)
	CHECK(e1432_auto_zero(hw, group));

    return 0;
}

static int
check_trailer(struct e1432_trailer *trailer, double clock_freq,
	      double span, int chan, int type)
{
    double  tmp;
    int     df2, df5;

    if (trailer->zoom_corr != 0)
    {
	/* Zoom correction is not currently implemented */
	(void) fprintf(stderr,
		       "%s: trailer zoom corr non-zero: %g (0x%lx)\n",
		       progname, trailer->zoom_corr,
		       *(long *) &trailer->zoom_corr);
	return -1;
    }
    if (trailer->gap < 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer gap negative: 0x%lx\n",
		       progname, trailer->gap);
	return -1;
    }
    if (trailer->rpm1 != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer rpm1 non-zero: %g (0x%lx)\n",
		       progname, trailer->rpm1,
		       *(long *) &trailer->rpm1);
	return -1;
    }
    if (trailer->rpm2 != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer rpm2 non-zero: %g (0x%lx)\n",
		       progname, trailer->rpm2,
		       *(long *) &trailer->rpm2);
	return -1;
    }
    if (trailer->peak != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer peak non-zero: %g (0x%lx)\n",
		       progname, trailer->peak,
		       *(long *) &trailer->peak);
	return -1;
    }
    if (trailer->rms != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer rms non-zero: %g (0x%lx)\n",
		       progname, trailer->rms,
		       *(long *) &trailer->rms);
	return -1;
    }

    /* Compute df2 and df5 from clock_freq and span */
    tmp = span * 2.56;
    df2 = 0;
    df5 = 0;
    while (tmp < clock_freq * 0.9999)
    {
	df2++;
	tmp *= 2;
    }
    if (tmp > clock_freq * 1.0001)
    {
	tmp /= 8;
	tmp *= 5;
	df2 -= 3;
	df5++;
	if (tmp > clock_freq * 1.0001 || tmp < clock_freq * 0.9999)
	{
	    (void) fprintf(stderr,
			   "%s: invalid span/clock_freq combination: %g/%g\n",
			   progname, span, clock_freq);
	    return -1;
	}
    }

    if (df2 != ((trailer->info & E1432_TRAILER_INFO_DEC_2_MASK)
		>> E1432_TRAILER_INFO_DEC_2_SHIFT))
    {
	(void) fprintf(stderr,
		       "%s: trailer info df2 mismatch: 0x%8.8lx, %d\n",
		       progname, trailer->info, df2);
	return -1;
    }
    if (df5 != ((trailer->info & E1432_TRAILER_INFO_DEC_5) != 0))
    {
	(void) fprintf(stderr,
		       "%s: trailer info df5 mismatch: 0x%8.8lx, %d\n",
		       progname, trailer->info, df5);
	return -1;
    }

    if (((trailer->info & E1432_TRAILER_INFO_CHAN_MASK) >>
	 E1432_TRAILER_INFO_CHAN_SHIFT) != chan - 1)
    {
	(void) fprintf(stderr,
		       "%s: trailer info chan mismatch: 0x%8.8lx, 0x%x\n",
		       progname, trailer->info, chan - 1);
	return -1;
    }
    if (((trailer->info & E1432_TRAILER_INFO_TYPE_MASK) >>
	 E1432_TRAILER_INFO_TYPE_SHIFT) != type)
    {
	(void) fprintf(stderr,
		       "%s: trailer info type mismatch: 0x%8.8lx, 0x%x\n",
		       progname, trailer->info, type);
	return -1;
    }

    return 0;
}

static int
wait_block_avail(E1432ID hw, int group, int scan, int verbose,
		 long blocksize, double span) 
{
    clock_t start, timeout;
    int     status;

    timeout = (2 + 2 * (blocksize / (span * 2.56))) * CLOCKS_PER_SEC;
    if (verbose > 2)
	(void) printf("Waiting %g sec for block available\n",
		      (double) timeout / CLOCKS_PER_SEC);
    start = clock();
    while ((status = e1432_block_available(hw, group)) == 0)
	if (clock() - start > timeout &&
	    (status = e1432_block_available(hw, group)) == 0)
	{
	    (void) fprintf(stderr, "%s: e1432_block_available: "
			   "timeout waiting %g sec\n",
			   progname, (double) timeout / CLOCKS_PER_SEC);
	    return -1;
	}
    if (status < 0)
    {
	(void) fprintf(stderr,
		       "%s: e1432_block_available: returned %d\n",
		       progname, status);
	return -1;
    }
    if (verbose > 0)
	(void) printf("Scan %d block available found\n", scan);

    return 0;
}

/*ARGSUSED*/
static int
run(E1432ID hw, int group, int nchan, SHORTSIZ16 *chan_list,
    int all_group, double clock_freq, double span, int verbose)
{
    FLOATSIZ64 buffer[BLOCKSIZE];
    LONGSIZ32 count;
    struct e1432_trailer trailer;
    int     scan;

    CHECK(e1432_init_measure(hw, all_group));

    for (scan = 0; scan < 3; scan++)
    {
	/* Wait for block available */
	CHECK(wait_block_avail(hw, E1432_INPUT_CHAN(1),
			       scan, verbose, BLOCKSIZE, span));

	/* Read the data */
	CHECK(e1432_read_float64_data(hw, E1432_INPUT_CHAN(1),
				      E1432_TIME_DATA, buffer,
				      BLOCKSIZE, &trailer, &count));
	if (count != BLOCKSIZE)
	{
	    (void) fprintf(stderr,
			   "%s: e1432_read_float64_data: "
			   "actual count was %ld, expected %d\n",
			   progname, count, BLOCKSIZE);
	    return -1;
	}

	CHECK(check_trailer(&trailer, clock_freq, span,
			    E1432_INPUT_CHAN(1), 0));
    }

    return 0;
}

static void
usage(void)
{
    (void) fprintf(stderr,
		   "Usage: %s [-uvVz] [-c clock_freq] [-L laddr]\n"
		   "\t[-m src_mode] [-n nchan] [-N nmod] [-s span]\n"
		   "\t-c: Set clock_freq to <clock_freq>, default 65536\n"
		   "\t-L: First logical address is <laddr>, default 8\n"
		   "\t-m: Set source mode to <src_mode>, default BRAND\n"
		   "\t-n: Use <nchan> channels, default all\n"
		   "\t-N: Use <nmod> modules, default 1\n"
		   "\t-s: Set span to <span>, default 12800\n"
		   "\t-u: Print this usage message\n"
		   "\t-v: Verbose output\n"
		   "\t-V: Print version info\n"
		   "\t-z: Do source autozero\n",
		   progname);
    exit(2);
}

int
main(int argc, char **argv)
{
    E1432ID hw;
    SHORTSIZ16 laddr[NMOD_MAX];
    SHORTSIZ16 chan_list[NCHAN_MAX];
    double  clock_freq, span;
    char   *p;
    int     c, i, j, k, auto_zero, nmod, verbose, src_mode;
    int     group, all_group, nchan;

    /* Get program name */
    progname = strrchr(argv[0], '/');
    if (progname == NULL)
	progname = argv[0];
    else
	progname++;

    /* Set option defaults */
    auto_zero = 0;
    clock_freq = 65536;
    laddr[0] = 8;
    src_mode = E1432_SOURCE_MODE_BRAND;
    nchan = -1;			/* Meaning use all channels */
    nmod = 1;
    span = 12800;
    verbose = 0;

    /* Process command-line options */
    while ((c = getopt(argc, argv, "c:L:m:n:N:s:uvVz")) != -1)
	switch (c)
	{
	case 'c':
	    if (sscanf(optarg, "%lg", &clock_freq) != 1)
	    {
		(void) fprintf(stderr,
			       "%s: invalid clock_freq: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'L':
	    laddr[0] = (SHORTSIZ16) strtol(optarg, &p, 0);
	    if (optarg == p || laddr[0] < 0 || laddr[0] > 255)
	    {
		(void) fprintf(stderr,
			       "%s: invalid logical address: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'm':
	    p = optarg;
	    while (*p != '\0')
	    {
		*p = tolower(*p);
		p++;
	    }
	    if (strcmp(optarg, "bsine") == 0)
		src_mode = E1432_SOURCE_MODE_BSINE;
	    else if (strcmp(optarg, "brand") == 0)
		src_mode = E1432_SOURCE_MODE_BRAND;
	    else
	    {
		(void) fprintf(stderr,
			       "%s: invalid source mode: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'n':
	    nchan = strtol(optarg, &p, 0);
	    if (optarg == p || nchan < -1 || nchan > NCHAN_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of channels: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'N':
	    nmod = strtol(optarg, &p, 0);
	    if (optarg == p || nmod < 0 || nmod > NMOD_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of modules: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 's':
	    if (sscanf(optarg, "%lg", &span) != 1)
	    {
		(void) fprintf(stderr,
			       "%s: invalid span: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'v':
	    verbose++;
	    break;
	case 'V':
	    (void) printf("%s\n", rcsid);
	    exit(EXIT_SUCCESS);
	case 'z':
	    auto_zero = 1;
	    break;
	case 'u':
	default:
	    usage();
	}

    if (argc > optind)
    {
	(void) fprintf(stderr, "%s: extra command-line arguments\n",
		       progname);
	usage();
    }

    /* Assume logical addresses are consecutive */
    for (i = 1; i < nmod; i++)
	laddr[i] = laddr[i - 1] + 1;

    /* Run the measurement */
    for (k = 0; k < 3; k++)
    {
	if (init(nmod, laddr, &hw, &group, &nchan,
		 chan_list, &all_group) < 0)
	    return EXIT_FAILURE;
	if (nchan < 1)
	{
	    (void) fprintf(stderr, "%s: need at least one source channel\n",
			   progname);
	    return EXIT_FAILURE;
	}
	for (j = 0; j < 3; j++)
	{
	    if (setup(hw, group, nchan, chan_list, all_group,
		      auto_zero, clock_freq, src_mode, span) < 0)
		return EXIT_FAILURE;

	    for (i = 0; i < 3; i++)
		if (run(hw, group, nchan, chan_list,
			all_group, clock_freq, span, verbose) < 0)
		    return EXIT_FAILURE;
	}
    }

    return EXIT_SUCCESS;
}
